
/*
 * Simple Watch 3 with Flappy Birdz
 *
 * A simple watch program with added Flappy Birdz game from Tiny Circuits
 *
 * Demonstrates the use of the Arduino Time library to make a simple digital watch
 * In this we write functions to set time and date and learn how to use the buttons
 * Also learn Arduino 'switch' and 'for' statements/commands
 * 
 * Uses Arduino Time library http://playground.arduino.cc/code/time  
 * Maintained by Paul Stoffregen https://github.com/PaulStoffregen/Time
 *
 * This code is for the TinyScreen+ board by TinyCircuits used by O Watch http://tiny-circuits.com 
 *
 * This example is created by O Watch on 15 March 2016 http://theowatch.com 
 * 
 * //-------------------------------------------------------------------------------
//  TinyCircuits TinyScreen FlappyBirdz Game Example
//  Last Updated 14 March 2016
//
//  This example shows how to create simple sprites with a memory structure ts_sprite
//  and add them to a list (spriteList) allowing for relatively simple game mechanics
//  in the main loop by modifying sprite x/y coordinates, checking for pixel collisions
//  using testPixelCollision, and displaying everything through drawBuffer.
//
//  Written by Ben Rose for TinyCircuits, http://TinyCircuits.com
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//-------------------------------------------------------------------------------
 * 
 * 
 * Button Actions:
 * ==============
 * 
 * Top-Right: Shows Time for 10s
 * Top-Left: Enters Flappy Bird Game
 * Bottom-Right: Set Date Mode
 * Bottom-Left: Set Time Mode
 * 
 * Inside Game, press Bottom-Left button to exit game
 * Inside Time/Date set mode, press Top-Left button to save & exit each step
 * 
 * 
*/

//ALPHA is just a value not equal to another color we're using- then we can test for ALPHA when needed.
const uint16_t ALPHA = 0x1111;

#include <TinyScreen.h> //include TinyScreen library
#include <TimeLib.h> //include the Arduino Time library.h
#include "flappyBirdSprites.h" //Add the flappy bird 'sprites'

TinyScreen display = TinyScreen(TinyScreenPlus); //Create the TinyScreen object

// Declare Flappy Birdz variables

typedef struct
{
  int x;
  int y;
  int width;
  int height;
  int bitmapNum;
  const unsigned int * bitmap;

} ts_sprite;

typedef struct
{
  int x;
  int y;
  int height;
  char stringChars[40];

} ts_string;

bool testBitmapCollision(ts_sprite *s1, ts_sprite *s2);
bool testPixelCollision(ts_sprite *s1, ts_sprite *s2);

int collisionDetected = 0;
int wingChangeCount = 3;
int wingPos = 1;
int pipeOffsetX = 5;
int pipeSpacingX = 34;
int movePipe = 4;
int movePipeMod = 1;
int pipeSpacingY = 32;
int frame = 0;
int cloudSpacingX = 30;
int cloudOffsetX = 0;
int speedUpBoxActive = 0;
int speedUpBoxHit = 0;
int speedUp = 0;
int slowBoxActive = 0;
int slowBoxHit = 0;
int slowUp = 0;
int closeBoxActive = 0;
int closeBoxHit = 0;
int doCloseBox = 0;
int darkBoxActive = 0;
int darkBoxHit = 0;
int doDarkBox = 0;
int heartBoxActive = 0;
int heartBoxHit = 0;

int defaultBrightness = 8;
int currentBrightness = 8;
int startScreen = 1;


ts_sprite flappyBird = { -25, 22, 17, 12, 0, flappyBirdBitmap};
ts_sprite wing = { -25, 23, 7, 8, 0, wingBitmap};
ts_sprite pipeUp0 = {0, 40, 12, 40, 0, greenPipeUpBitmap};
ts_sprite pipeUp1 = {0, 40, 12, 40, 0, greenPipeUpBitmap};
ts_sprite pipeUp2 = {0, 40, 12, 40, 0, greenPipeUpBitmap};
ts_sprite pipeUp3 = {0, 40, 12, 40, 0, greenPipeUpBitmap};
ts_sprite pipeDown0 = {0, -pipeSpacingY, 12, 40, 0, greenPipeDownBitmap};
ts_sprite pipeDown1 = {0, -pipeSpacingY, 12, 40, 0, greenPipeDownBitmap};
ts_sprite pipeDown2 = {0, -pipeSpacingY, 12, 40, 0, greenPipeDownBitmap};
ts_sprite pipeDown3 = {0, -pipeSpacingY, 12, 40, 0, greenPipeDownBitmap};
ts_sprite cloud0 = {55, 0, 15, 10, 0, cloudBitmap};
ts_sprite cloud1 = {55, 15, 15, 10, 0, cloudBitmap};
ts_sprite cloud2 = {55, 30, 15, 10, 0, cloudBitmap};
ts_sprite cloud3 = {55, 30, 15, 10, 0, cloudBitmap};
ts_sprite ground = {0, 52, 105, 12, 0, groundBitmap};
ts_sprite speedBox = { -10, -10, 10, 10, 0, speedUpBoxBitmap};
ts_sprite closeBox = { -10, -10, 10, 10, 0, closeBoxBitmap};
ts_sprite darkBox = { -10, -10, 10, 10, 0, darkBoxBitmap};
ts_sprite slowBox = { -10, -10, 10, 10, 0, slowBoxBitmap};
ts_sprite heartBox = { -10, -10, 10, 10, 0, heartBoxBitmap};
ts_sprite title = { 0, -60, 96, 56, 0, flappyTitle};
ts_sprite hearts = { -35, 2, 70, 5, 0, heartBitmap};

int lives = 0;
ts_sprite * spriteList[22] = {&cloud0, &cloud1, &cloud2, &cloud3, &pipeUp0, &pipeDown0, &pipeUp1, &pipeDown1, &pipeUp2, &pipeDown2, &pipeUp3, &pipeDown3, &ground, &hearts, &closeBox, &speedBox, &darkBox, &heartBox, &slowBox, &flappyBird, &wing, &title};

int amtSprites = 22;

int highScore = 0;
int currentScore = 0;
int showScore = 0;
ts_string score = {0, -20, 0, "0"};

// End flappy birz variables


void setup()
{
  display.begin();                            //Initializes TinyScreen board
  display.setFlip(1);                         //Flips the TinyScreen rightside up for O Watch
  display.on();                               //Turns TinyScreen display on
  display.fontColor(TS_8b_White,TS_8b_Black); //Set the font color, font background
  display.setBrightness(10);                  //Set display brightness 0 - 15
  
  // Set the time and date. Change this to your current date and time.
  setTime(7,34,55,14,3,2016);    //values in the order hr,min,sec,day,month,year

  //Show info at start to say what each button does
  display.setFont(liberationSansNarrow_14ptFontInfo);
  display.setCursor(35,30);
  display.print("Set");
  display.setCursor(32,0);
  display.print("Show");  
  display.setFont(liberationSansNarrow_8ptFontInfo);
  display.setCursor(1,15);
  display.print("<Game");  
  display.setCursor(70,15);
  display.print("Time>");
  display.setCursor(60,45);
  display.print("Date>");  
  display.setCursor(5,45);
  display.print("<Time");
  delay(8000);
  display.clearScreen();
  display.off();
}

//function to read button press
int getInput() {
  if (display.getButtons())
    return 1;

  return 0;
}


void loop()
{
  display.begin(); //reinitialize TS since bitdepth is changed within loop for game
  //Switch statement to select part of code to run based on button selection
  switch (display.getButtons()) 
  {

    //Run updateTime function 
    case TSButtonLowerLeft:
      display.on();
      updateTime();
      delay(200);
      display.off();
      break;

    //Run updateDate function
    case TSButtonLowerRight:
      delay(200);
      display.on();
      updateDate();
      delay(200);
      display.off();
      break;

    //Show time for 10 seconds 
    case TSButtonUpperRight:
      display.on();
      for(int i=0; i<11; i++)
      {
        display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
        // Print date in US format MM:DD:YY (Switch the order in which day, month, year that you like to use)
        display.setCursor(15,8); //Set the cursor where you want to start printing the date
        display.print(monthShortStr(month()));
        display.print(" ");
        display.print(day()); 
        display.print(", ");
        display.print(year());
      
        display.setCursor(30,25); //Set the cursor where you want to start printing the date  
        display.print(dayStr(weekday()));
      
        display.setFont(liberationSansNarrow_16ptFontInfo);   //Set the font type
      
        // display time in HH:MM:SS 24 hour format
        display.setCursor(20,45); //Set the cursor where you want to start printing the time
        if(hour()<10) display.print(0); //print a leading 0 if hour value is less than 0
        display.print(hour());
        display.print(":");
        if(minute()<10) display.print(0); //print a leading 0 if minute value is less than 0
        display.print(minute());
        display.print(":");
        if(second()<10) display.print(0); //print a leading 0 if seconds value is less than 0
        display.print(second());
        display.print(" "); //just a empty space after the seconds
        delay(1000); //delay 1 second           
      }
      delay(200);
      display.clearScreen();
      display.off();
      break;


      //Flappy Birdz
      case TSButtonUpperLeft:
      display.setBitDepth(1);
      display.on();
      playFlappyBirdz(); 
      display.clearScreen();
      delay(1000);
      display.off();
      display.setBitDepth(0);
      break;
      

    
  }
  delay(200);
}


//Function to update time
void updateTime()
{

  //Set hour
  display.clearScreen();
  display.setFont(liberationSansNarrow_8ptFontInfo);   //Set the font type
  display.setCursor(2,15);
  display.print("<Save | Change>");
  display.setCursor(15,40);
  display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
  display.print("Set Hour:");
  int myhour=hour();
  while(!display.getButtons(TSButtonUpperLeft))
  {
    display.setCursor(75,40);
    if(myhour<10) display.print("0");
    display.print(myhour);
    if(display.getButtons(TSButtonUpperRight))
      myhour++;
    if(display.getButtons(TSButtonLowerRight))
      myhour--;
    if(myhour<0) myhour=24;
    if(myhour>24) myhour=0;
    delay(200);
  }
  setTime(myhour, minute(), second(), day(), month(), year());
  delay(500);
  display.clearScreen();
  display.setCursor(10,25);
  display.print("Hour Saved");
  delay(1000);

  
  //Set minute
  display.clearScreen();
  display.setFont(liberationSansNarrow_8ptFontInfo);   //Set the font type
  display.setCursor(2,15);
  display.print("<Save | Change>");
  display.setCursor(15,40);
  display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
  display.print("Set Mins:");
  int myminute=minute();
  while(!display.getButtons(TSButtonUpperLeft))
  {
    display.setCursor(75,40);
    if(myminute<10) display.print("0");
    display.print(myminute);
    if(display.getButtons(TSButtonUpperRight))
      myminute++;
    if(display.getButtons(TSButtonLowerRight))
      myminute--;
    if(myminute<0) myminute=60;
    if(myminute>60) myminute=0;
    delay(200);
  }
  setTime(hour(), myminute, second(), day(), month(), year());
  delay(500);
  display.clearScreen();
  display.setCursor(10,25);
  display.print("Minute Saved");
  delay(1000);

  //Set second
  display.clearScreen();
  display.setFont(liberationSansNarrow_8ptFontInfo);   //Set the font type
  display.setCursor(2,15);
  display.print("<Save | Change>");
  display.setCursor(15,40);
  display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
  display.print("Set Secs:");
  int mysecond=second();
  while(!display.getButtons(TSButtonUpperLeft))
  {
    display.setCursor(75,40);
    if(mysecond<10) display.print("0");
    display.print(mysecond);
    if(display.getButtons(TSButtonUpperRight))
      mysecond++;
    if(display.getButtons(TSButtonLowerRight))
      mysecond--;
    if(mysecond<0) mysecond=60;
    if(mysecond>60) mysecond=0;
    delay(200);
  }
  setTime(hour(), minute(), mysecond, day(), month(), year());
  delay(500);
  display.clearScreen();
  display.setCursor(10,25);
  display.print("Second Saved");
  delay(2000);
  display.clearScreen();
  
}

//Function to update date
void updateDate()
{

  //Set day
  display.clearScreen();
  display.setFont(liberationSansNarrow_8ptFontInfo);   //Set the font type
  display.setCursor(2,15);
  display.print("<Save | Change>");
  display.setCursor(10,40);
  display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
  display.print("Set Day:");
  int myday=day();
  while(!display.getButtons(TSButtonUpperLeft))
  {
    display.setCursor(75,40);
    if(myday<10) display.print("0");
    display.print(myday);
    if(display.getButtons(TSButtonUpperRight))
      myday++;
    if(display.getButtons(TSButtonLowerRight))
      myday--;
    if(myday<0) myday=31;
    if(myday>31) myday=0;
    delay(200);
  }
  setTime(hour(), minute(), second(), myday, month(), year());
  delay(500);
  display.clearScreen();
  display.setCursor(10,25);
  display.print("Day Saved");
  delay(1000);

  
  //Set month
  display.clearScreen();
  display.setFont(liberationSansNarrow_8ptFontInfo);   //Set the font type
  display.setCursor(2,15);
  display.print("<Save | Change>");
  display.setCursor(10,40);
  display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
  display.print("Set Month:");
  int mymonth=month();
  while(!display.getButtons(TSButtonUpperLeft))
  {
    display.setCursor(75,40);
    if(mymonth<10) display.print("0");
    display.print(mymonth);
    if(display.getButtons(TSButtonUpperRight))
      mymonth++;
    if(display.getButtons(TSButtonLowerRight))
      mymonth--;
    if(mymonth<0) mymonth=60;
    if(mymonth>60) mymonth=0;
    delay(200);
  }
  setTime(hour(), minute(), second(), day(), mymonth, year());
  delay(500);
  display.clearScreen();
  display.setCursor(10,25);
  display.print("Month Saved");
  delay(1000);

  //Set Year
  display.clearScreen();
  display.setFont(liberationSansNarrow_8ptFontInfo);   //Set the font type
  display.setCursor(2,15);
  display.print("<Save | Change>");
  display.setCursor(10,40);
  display.setFont(liberationSansNarrow_12ptFontInfo);   //Set the font type
  display.print("Set Year:");
  int myyear=year();
  while(!display.getButtons(TSButtonUpperLeft))
  {
    display.setCursor(68,40);
    if(myyear<10) display.print("0");
    display.print(myyear);
    if(display.getButtons(TSButtonUpperRight))
      myyear++;
    if(display.getButtons(TSButtonLowerRight))
      myyear--;
    if(myyear<2010) myyear=2030;
    if(myyear>2030) myyear=2010;
    delay(200);
  }
  setTime(hour(), minute(), second(), day(), month(), myyear);
  delay(500);
  display.clearScreen();
  display.setCursor(10,25);
  display.print("Year Saved");
  delay(1000);
  display.clearScreen();
  
}


//Flappy birdz functions



#define zmax(a,b) ((a)>(b)?(a):(b))
#define zmin(a,b) ((a)<(b)?(a):(b))

bool testBitmapCollision(ts_sprite *s1, ts_sprite *s2) {
  if (s1->x < s2->x + s2->width && s1->x + s1->width > s2->x)
    if (s2->y < s1->y + s1->height && s2->y + s2->height > s1->y)
      return true;
  return false;
}

bool testPixelCollision(ts_sprite *s1, ts_sprite *s2) {
  if (!testBitmapCollision(s1, s2))return false;
  int startX = zmax(s1->x, s2->x);
  int endX = zmin(s1->x + s1->width, s2->x + s2->width);
  int startY = zmax(s1->y, s2->y);
  int endY = zmin(s1->y + s1->height, s2->y + s2->height);
  for (int y = startY; y < endY; y++) {
    for (int x = startX; x < endX; x++) {
      if (s1->bitmap[(y - s1->y)*s1->width + (x - s1->x)] != ALPHA && s2->bitmap[(y - s2->y)*s2->width + (x - s2->x)] != ALPHA)
        return true;
    }
  }
  return false;
}

int writeCount = 0;

void drawBuffer() {
  uint8_t lineBuffer[96 * 64 * 2];
  display.startData();
  for (int y = 0; y < 64; y++) {
    for (int b = 0; b < 96; b++) {
      lineBuffer[b * 2] = TS_16b_Blue >> 8;
      lineBuffer[b * 2 + 1] = TS_16b_Blue;
    }
    for (int spriteIndex = 0; spriteIndex < amtSprites; spriteIndex++) {
      ts_sprite *cs = spriteList[spriteIndex];
      if (y >= cs->y && y < cs->y + cs->height) {
        int endX = cs->x + cs->width;
        if (cs->x < 96 && endX > 0) {
          int bitmapNumOffset = cs->bitmapNum * cs->width * cs->height;
          int xBitmapOffset = 0;
          int xStart = 0;
          if (cs->x < 0)xBitmapOffset -= cs->x;
          if (cs->x > 0)xStart = cs->x;
          int yBitmapOffset = (y - cs->y) * cs->width;
          for (int x = xStart; x < endX; x++) {
            unsigned int color = cs->bitmap[bitmapNumOffset + xBitmapOffset + yBitmapOffset++];
            if (color != ALPHA) {
              lineBuffer[(x) * 2] = color >> 8;
              lineBuffer[(x) * 2 + 1] = color;
            }
          }
        }
      }
    }
    putString(y, score.x, score.y, score.stringChars, lineBuffer, liberationSans_16ptFontInfo);
    if (startScreen) {
      putString(y, 1, 38, score.stringChars, lineBuffer, liberationSans_10ptFontInfo);
      char hs[10];
      sprintf(hs, "%d", highScore);
      putString(y, 74, 38, hs, lineBuffer, liberationSans_10ptFontInfo);

    }
    display.writeBuffer(lineBuffer, 96 * 2);
    
  }


  display.endTransfer();
}


void putString(int y, int fontX, int fontY, char * string, uint8_t * buff, const FONT_INFO& fontInfo) {
  const FONT_CHAR_INFO* fontDescriptor = fontInfo.charDesc;
  int fontHeight = fontInfo.height;
  if (y >= fontY && y < fontY + fontHeight) {
    const unsigned char* fontBitmap = fontInfo.bitmap;
    int fontFirstCh = fontInfo.startCh;
    int fontLastCh = fontInfo.endCh;
    //if(!_fontFirstCh)return 1;
    //if(ch<_fontFirstCh || ch>_fontLastCh)return 1;
    //if(_cursorX>xMax || _cursorY>yMax)return 1;
    int stringChar = 0;
    int ch = string[stringChar++];
    while (ch) {
      uint8_t chWidth = pgm_read_byte(&fontDescriptor[ch - fontFirstCh].width);
      int bytesPerRow = chWidth / 8;
      if (chWidth > bytesPerRow * 8)
        bytesPerRow++;
      unsigned int offset = pgm_read_word(&fontDescriptor[ch - fontFirstCh].offset) + (bytesPerRow * fontHeight) - 1;

      for (uint8_t byte = 0; byte < bytesPerRow; byte++) {
        uint8_t data = pgm_read_byte(fontBitmap + offset - (y - fontY) - ((bytesPerRow - byte - 1) * fontHeight));
        uint8_t bits = byte * 8;
        for (int i = 0; i < 8 && (bits + i) < chWidth; i++) {
          if (data & (0x80 >> i)) {
            buff[(fontX) * 2] = TS_16b_Yellow >> 8;
            buff[(fontX) * 2 + 1] = TS_16b_Yellow;
            // setPixelInBuff(y,16+fontX,0);
            //lineBuffer[16+fontX]=0;
          } else {
            //SPDR=_fontBGcolor;
          }
          fontX++;
        }
      }
      fontX += 2;
      ch = string[stringChar++];
    }
  }
}


//Flappy Birdz main code

void playFlappyBirdz()
{
  while(!display.getButtons(TSButtonLowerLeft)) //exit FB when you press LowerLeft button
      {
      frame++;
    
      ground.x--;
      if (ground.x < -4)ground.x = 0;
    
      if (pipeUp0.x + pipeUp0.width < 0) {
        pipeUp0.x = pipeUp1.x;
        pipeUp0.y = pipeUp1.y;
        pipeDown0.x = pipeDown1.x;
        pipeDown0.y = pipeDown1.y;
        pipeUp1.x = pipeUp2.x;
        pipeUp1.y = pipeUp2.y;
        pipeDown1.x = pipeDown2.x;
        pipeDown1.y = pipeDown2.y;
        pipeUp2.x = pipeUp3.x;
        pipeUp2.y = pipeUp3.y;
        pipeDown2.x = pipeDown3.x;
        pipeDown2.y = pipeDown3.y;
        pipeUp3.x = pipeUp2.x + pipeSpacingX - doCloseBox;
        pipeDown3.x = pipeDown2.x + pipeSpacingX - doCloseBox;
        pipeUp3.y = 33 + micros() % 18;
        pipeDown3.y = pipeUp3.y - pipeSpacingY - 40;
        doCloseBox = 0;
      }
      pipeUp0.x--;
      pipeDown0.x--;
      pipeUp1.x--;
      pipeDown1.x--;
      pipeUp2.x--;
      pipeDown2.x--;
      pipeUp3.x--;
      pipeDown3.x--;
    
      if (pipeUp3.y == 38 && !speedUpBoxActive) {
        speedUpBoxActive = 1;
        speedBox.x = pipeUp3.x + 2;
        speedBox.y = pipeUp3.y - (pipeSpacingY / 2) - (speedBox.height / 2);
        speedBox.bitmapNum = 0;
      } else if (pipeUp3.y == 37 && !closeBoxActive) {
        closeBoxActive = 1;
        closeBox.x = pipeUp3.x + 2;
        closeBox.y = pipeUp3.y - (pipeSpacingY / 2) - (closeBox.height / 2);
        closeBox.bitmapNum = 0;
      } else if (pipeUp3.y == 36 && !darkBoxActive) {
        darkBoxActive = 1;
        darkBox.x = pipeUp3.x + 2;
        darkBox.y = pipeUp3.y - (pipeSpacingY / 2) - (darkBox.height / 2);
        darkBox.bitmapNum = 0;
      } else if (pipeUp3.y == 35 && !slowBoxActive) {
        slowBoxActive = 1;
        slowBox.x = pipeUp3.x + 2;
        slowBox.y = pipeUp3.y - (pipeSpacingY / 2) - (slowBox.height / 2);
        slowBox.bitmapNum = 0;
      } else if (pipeUp3.y == 34 && !heartBoxActive) {
        heartBoxActive = 1;
        heartBox.x = pipeUp3.x + 2;
        heartBox.y = pipeUp3.y - (pipeSpacingY / 2) - (heartBox.height / 2);
        heartBox.bitmapNum = 0;
      }
    
      if (frame & 1)
        cloudOffsetX--;
      if (cloudOffsetX <= -14) {
        cloudOffsetX += cloudSpacingX;
        cloud0.y = cloud1.y;
        cloud1.y = cloud2.y;
        cloud2.y = cloud3.y;
        cloud3.y = -5 + micros() % 40;
      }
      cloud0.x = cloudOffsetX + cloudSpacingX * 0;
      cloud1.x = cloudOffsetX + cloudSpacingX * 1;
      cloud2.x = cloudOffsetX + cloudSpacingX * 2;
      cloud3.x = cloudOffsetX + cloudSpacingX * 3;
    
      if (speedUpBoxActive) {
        speedBox.x--;
        if (speedUpBoxHit) speedBox.y--;
        if (speedBox.x < -speedBox.width || speedBox.y < -speedBox.height) {
          speedUpBoxHit = 0;
          speedUpBoxActive = 0;
        }
        if (!(frame % 3)) {
          speedBox.bitmapNum ^= 1;
        }
        if (testPixelCollision(&flappyBird, &speedBox)) {
          speedUpBoxHit = 1;
          speedUp = 10000;
        }
      }
    
      if (slowBoxActive) {
        slowBox.x--;
        if (slowBoxHit) slowBox.y--;
        if (slowBox.x < -slowBox.width || slowBox.y < -slowBox.height) {
          slowBoxHit = 0;
          slowBoxActive = 0;
        }
        if (!(frame % 3)) {
          slowBox.bitmapNum ^= 1;
        }
        if (testPixelCollision(&flappyBird, &slowBox)) {
          slowBoxHit = 1;
          speedUp = -10000;
        }
      }
    
      if (closeBoxActive) {
        closeBox.x--;
        if (closeBoxHit) closeBox.y--;
        if (closeBox.x < -closeBox.width || closeBox.y < -closeBox.height) {
          closeBoxHit = 0;
          closeBoxActive = 0;
        }
        if (frame & 1) {
          closeBox.bitmapNum ^= 1;
        }
        if (testPixelCollision(&flappyBird, &closeBox)) {
          closeBoxHit = 1;
          doCloseBox = 10;
        }
      }
    
      if (darkBoxActive) {
        darkBox.x--;
        if (darkBoxHit) darkBox.y--;
        if (darkBox.x < -darkBox.width || darkBox.y < -darkBox.height) {
          darkBoxHit = 0;
          darkBoxActive = 0;
        }
        if (frame & 1) {
          darkBox.bitmapNum ^= 1;
        }
        if (testPixelCollision(&flappyBird, &darkBox)) {
          darkBoxHit = 1;
          doDarkBox = 20;
        }
      }
    
      if (heartBoxActive) {
        heartBox.x--;
        if (heartBoxHit) heartBox.y--;
        if (heartBox.x < -heartBox.width || heartBox.y < -heartBox.height) {
          heartBoxHit = 0;
          heartBoxActive = 0;
        }
        if (frame & 1) {
          heartBox.bitmapNum ^= 1;
        }
        if (!heartBoxHit && testPixelCollision(&flappyBird, &heartBox)) {
          heartBoxHit = 1;
          lives++;
        }
      }
    
      if (getInput()) {
        wingChangeCount = 2;
        if (flappyBird.y > 0) {
          flappyBird.y -= 2;
          wing.y = flappyBird.y + 1;
        }
      } else {
        wingChangeCount = 6;
        if (flappyBird.y < ground.y - flappyBird.height) {
          flappyBird.y += 1;
          wing.y = flappyBird.y + 1;
        }
      }
    
      if (!(frame % wingChangeCount)) {
        wing.bitmapNum++;
        if (wing.bitmapNum > 2)wing.bitmapNum = 0;
      }
    
      pipeUp1.bitmap = greenPipeUpBitmap;
      pipeDown1.bitmap = greenPipeDownBitmap;
      pipeUp0.bitmap = greenPipeUpBitmap;
      pipeDown0.bitmap = greenPipeDownBitmap;
    
      if (testPixelCollision(&flappyBird, &pipeUp0)) {
        pipeUp0.bitmap = redPipeUpBitmap;
        if (!collisionDetected) {
          collisionDetected = 1;
          lives--;
        }
      } else if (testPixelCollision(&flappyBird, &pipeUp1)) {
        pipeUp1.bitmap = redPipeUpBitmap;
        if (!collisionDetected) {
          collisionDetected = 1;
          lives--;
        }
      } else if (testPixelCollision(&flappyBird, &pipeDown0)) {
        pipeDown0.bitmap = redPipeDownBitmap;
        if (!collisionDetected) {
          collisionDetected = 1;
          lives--;
        }
      } else if (testPixelCollision(&flappyBird, &pipeDown1)) {
        pipeDown1.bitmap = redPipeDownBitmap;
        if (!collisionDetected) {
          collisionDetected = 1;
          lives--;
        }
      }
    
      if (pipeUp0.x + pipeUp0.width == flappyBird.x) {
        if (!collisionDetected) {
          currentScore++;
          if (!(currentScore % 10)) {
            lives++;
          }
          sprintf(score.stringChars, "%d", currentScore);
          score.x = 50;
          score.y = 20;
          showScore = 1;
        }
        collisionDetected = 0;
      }
    
      if (hearts.x > (lives * 7) - hearts.width + 2) {
        hearts.x--;
      }
      if (hearts.x < (lives * 7) - hearts.width + 2) {
        hearts.x++;
      }
    
      if (!lives) {
        if (currentScore > highScore) {
          highScore = currentScore;
        }
        startScreen = 1;
        currentScore = 0;
      }
    
      if (showScore) {
        if (score.y > -20)
          score.y -= 2;
        else
          showScore = 0;
      }
    
      if (startScreen) {
        if (flappyBird.x > -25) {
          flappyBird.x--;
          wing.x--;
        }
        if (title.y < 5)
          title.y += 2;
        else if (getInput()) {
          startScreen = 0;
          title.y = -60;
          flappyBird.x = 25;
          wing.x = 25;
          flappyBird.y = (pipeUp0.y + pipeUp1.y) / 2 - (pipeSpacingY / 2) - (flappyBird.height / 3);
          wing.y = flappyBird.y + 1;
          lives = 3;
        }
      }
    
    
    
    
      unsigned long timer = micros();
      drawBuffer();
      timer = micros() - timer;
      //while(!SerialUSB);
      //SerialUSB.println(timer);
      //delay(100);
      int delayTime = 14000 - speedUp - (currentScore / 10) * 1000;
      if (delayTime < 0)delayTime = 0;
      delayMicroseconds(delayTime);
      if (speedUp > 0)speedUp -= 50;
      if (speedUp < 0)speedUp += 100;
    
      if (!(frame % 4)) {
        if (doDarkBox) {
          doDarkBox--;
          currentBrightness--;
          if (currentBrightness < 0)currentBrightness = 0;
          display.setBrightness(currentBrightness);
        } else if (currentBrightness < defaultBrightness) {
          currentBrightness++;
          display.setBrightness(currentBrightness);
        }
      }

      }
}

